/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_10;

import djbuch.kapitel_06.GC;
import lotus.domino.*;

/**
 * 
 * Demo Klasse, die die Verhaltensweise der verschiedenen create Methoden fr 
 * ViewNavigator in View demonstrieren.
 * 
 * @author Thomas Ekert
 */
public class ViewNavigatorDemo extends NotesThread {

	private static final String PASSWORD = "geheim";
	private static final String HOST = null;//"www.djbuch.de"; 		//null fr lokale Session
	private static final String USER = null;//"Thomas Ekert/DJBUCH";	//null fr lokale Session

	/** 
	 * Startet die verschiedenen Demos fr createViewNav, createViewNavMaxLevel, 
	 * createViewNavFrom, createViewNavFromChildren, createViewNavFromDescendants
	 * und createViewNavFromCategory
	 */

	public static void main(String[] args) {
		ViewNavigatorDemo vnd = new ViewNavigatorDemo();
		vnd.start();
	}
	public void runNotes() {
		View view = null;
		Session session = null;
		ViewNavigator viewNav = null;
		try {
			session = NotesFactory.createSession(HOST, USER, PASSWORD);
			db = session.getDatabase(session.getServerName(), "djbuch/djbuch.nsf");
			view=db.getView ("V_viewNavigator_k10");
			setup();
			//Demo 1 - Gesamte Ansicht per createViewNav
			System.out.println (LINE+"Gesamte Ansicht: ");
			view.refresh();
			viewNav = view.createViewNav();
			printNavigator(viewNav);
			viewNav = GC.recycle (viewNav);
			System.out.println ("... done ... \n\n");
			//Demo 2 - Gesamte Ansicht, aber nur bis zu einer bestimmten Tiefe per createViewNavMaxLevel
			for (int i = 0; i < 3; i++) {
				System.out.println (LINE+"Nur bis Level " + i);
				viewNav = view.createViewNavMaxLevel(i);
				printNavigator(viewNav);
				viewNav=GC.recycle (viewNav);
			}
			System.out.println ("... done ... \n\n");
			
			//Demo 3 - 	Teil einer Ansicht unterhalb eines bestimmten Eintrages
			// per createViewNavFrom
			demoChildrenDescendants ("createViewNavFrom", view,viewNav,TYPE_FROM);
			//Demo 4 - Teil einer Ansicht unterhalb eines ViewEntrys - Nur Children
			// per createViewNavFromChildren
			demoChildrenDescendants ("createViewNavFromChildren", view,viewNav,TYPE_CHILDREN);
			//Demo 5 - Teil einer Ansicht unterhalb eines ViewEntrys - Nur Children
			// per createViewNavFromDescendants
			demoChildrenDescendants ("createViewNavFromDescendants", view,viewNav,TYPE_DESCENDANTS);
			
			//Demo 6 - View Navigator anhand eines Kategorienamens erstellen
			for (int i = 0; i < SUB_CATEGORIE.length; i++) {
				System.out.println (LINE+"Navigator der Kategorie " + SUB_CATEGORIE[i]);
				viewNav = view.createViewNavFromCategory(SUB_CATEGORIE[i]);
				printNavigator(viewNav);
				viewNav=GC.recycle (viewNav);
			}
			System.out.println ("... done ... \n\n");
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			GC.recycle (viewNav);
			GC.recycle (view);
			GC.recycle (db);
			GC.recycle (session);
		}
	}
	
	/**
	 * Rumt zunchst alle Dokumente auf, die zuvor in Tests erstellt wurden.
	 * Erstellt Dokumente mit einer hierarchischen Anordnung ber ein Kategoriefeld.
	 */
	private static void setup() {
		Document doc = null;
		Document respDoc = null;
		try {
			cleanup();
			doc = createDoc ("Erste Kategorie", "1. Erstes Dokument"); GC.recycle (doc);
			doc = createDoc ("Erste Kategorie", "2. Zweites Dokument");
			respDoc = createDoc ("Andere Kategorie", "Antwort auf Zweites Dokument", doc);
			GC.recycle (doc); GC.recycle (respDoc);
			doc = createDoc ("Zweite Kategorie\\Sub eins", "3. Drittes Dokument"); GC.recycle (doc);
			doc = createDoc ("Zweite Kategorie\\Sub zwei", "4. Viertes Dokument"); GC.recycle (doc);
			doc = createDoc ("Zweite Kategorie\\Sub zwei", "5. Fnftes Dokument"); GC.recycle (doc);
			doc = createDoc ("Zweite Kategorie", "6. Sechstes Dokument"); GC.recycle (doc);
			doc = createDoc ("Zweite Kategorie", "7. Siebtes Dokument"); GC.recycle (doc);
			System.out.println ("Dokumente erzeugt.");
		} catch (NotesException e) {
			e.printStackTrace();
		} finally {
			GC.recycle (doc);
			GC.recycle (respDoc);
		}
	}

	/**
	 * Erstellt ein Dokument mit den Eigenschaften:
	 * @param cat - Item F_category == cat
	 * @param title - Item F_titel == title
	 * @param parent - Falls parent != null wird ein Antwortdokument zu parent erstellt.
	 * @return - Das neue Dokument
	 * @throws NotesException
	 */
	private static Document createDoc (String cat, String title, Document parent) throws NotesException {
		Document doc = db.createDocument();
		doc.replaceItemValue ("Form", "FO_dokument_k6");
		doc.replaceItemValue ("F_Category", cat);
		doc.replaceItemValue ("F_titel", title);
		//Anhand dieses Markers kann erkannt werden, dass createDoc ein Dokument erstellt hat.
		//Wird frs Aufrumen bentigt - siehe cleanup.
		doc.replaceItemValue ("F_marker_k6",new Integer (1));
		if (parent != null) {
			doc.makeResponse(parent);
		}
		doc.save (true,false);
		return doc;
	}
	
	private static Document createDoc (String cat, String title) throws NotesException {
		return createDoc (cat, title, null);
	}
	
	/**
	 * Einfache Ausgabe Routine fr ViewNavigator
	 * @param nav
	 */
	public static final void printNavigator (ViewNavigator nav) {
		ViewEntry entry = null, nextEntry = null;
		try {
			entry = nav.getFirst();
			while (entry != null) {
				if (entry.isDocument()) {
					System.out.println ("\tDokument: " + entry.getDocument().getItemValueString("F_titel"));
				}
				if (entry.isCategory()) {
					System.out.println ("Kategorie: " + entry.getColumnValues().elementAt(0));
				}
				if (entry.isTotal()) {
					System.out.println ("Total: " + entry.getColumnValues().elementAt(3));
				}
				if (entry.isConflict()) {
					System.out.println ("\t\tSpeicherkonflikt: " + entry.getDocument().getItemValueString("F_titel"));
				}
				nextEntry = nav.getNext();
				GC.recycle(entry);
				entry=nextEntry;
			}
		} catch (NotesException e) {
			e.printStackTrace();
		} finally {
			GC.recycle (entry);
			GC.recycle (nextEntry);
		}
	}
	
	/**
	 * Erstellt einen ViewNavigator und gibt diesen aus.
	 * Fr die Erstellung werden ViewEntries oder Dokumente bentigt, die anhand der
	 * Konstanten SUB_ENTRIES oder SUB_DOCS gefunden werden. Diese Arrays enthalten
	 * Titel von Dokumenten oder Namen von Kategorien.
	 * Die Demo loopt ber alle Eintrge in diesen Konstanten, um das unterschiedliche Verhalten 
	 * anzuzeigen.
	 * @param msg - Zustzlich auszugebende Info Message.
	 * @param view - View, auf dem der Navigator basiert.
	 * @param viewNav - ein (leeres) ViewNavigator Objekt
	 * @param type - gibt an, wie der ViewNavigator erstellt werden soll. Dies beeinflusst, ob beim 
	 * erstellen die Methode createViewNavFrom, createViewNavFromChildren
	 * oder createViewNavFromDescendants verwendet wird.
	 * @throws NotesException
	 */
	private static final void demoChildrenDescendants (String msg, View view, ViewNavigator viewNav, int type) throws NotesException {
		//Demo A - Getestete Elemente sind ViewEntries
		System.out.println (LINE + msg + " - Demo A");
		for (int i = 0; i < SUB_ENTRIES.length; i++) {
			switch (type) {
				case (TYPE_FROM):
					System.out.println (LINE+"Unterhalb von Entry " + SUB_ENTRIES[i]);
					viewNav = view.createViewNavFrom(getElement (view,SUB_ENTRIES[i]));
					break;
				case (TYPE_CHILDREN):
					System.out.println (LINE+"Children unterhalb von Entry " + SUB_ENTRIES[i]);
					viewNav = view.createViewNavFromChildren(getElement (view,SUB_ENTRIES[i]));
					break;
				case (TYPE_DESCENDANTS):
					System.out.println (LINE+"Descendants unterhalb von Entry " + SUB_ENTRIES[i]);
					viewNav = view.createViewNavFromDescendants(getElement (view,SUB_ENTRIES[i]));
					break;					
				default:
					throw new NotesException (999,"Fehlerhafter Type.");
			}
			printNavigator(viewNav);
			viewNav=GC.recycle (viewNav);
		}
		//Demo B - Getestete Elemente sind Documents
		System.out.println (LINE + msg + " - Demo B");
		for (int i = 0; i < SUB_DOCS.length; i++) {
			switch (type) {
				case (TYPE_FROM):
					System.out.println (LINE+"Unterhalb von Document " + SUB_DOCS[i]);
					viewNav = view.createViewNavFrom(getDocument (SUB_DOCS[i]));
					break;
				case (TYPE_CHILDREN):
					System.out.println (LINE+"Children unterhalb von Document " + SUB_DOCS[i]);
					viewNav = view.createViewNavFromChildren(getDocument (SUB_DOCS[i]));
					break;
				case (TYPE_DESCENDANTS):
					System.out.println (LINE+"Descendants unterhalb von Document " + SUB_DOCS[i]);
					viewNav = view.createViewNavFromDescendants(getDocument (SUB_DOCS[i]));
					break;					
				default:
					throw new NotesException (999,"Fehlerhafter Type.");
			}
			printNavigator(viewNav);
			viewNav=GC.recycle (viewNav);
		}
		System.out.println ("... done ... \n\n");
	}
	
	/**
	 * Ldt ein ViewEntry in einem View. Rckgabewert ist entweder eine Kategorie mit namen
	 * key oder ein Dokument fr das das Item F_titel den Wert key hat.
	 * @param view - Ansicht in der gesucht werden soll.
	 * @param key - Schlssel nach dem gesucht wird (Kategorie oder Dokumententitel)
	 * @return
	 */
	private static final ViewEntry getElement (View view, String key) {
		ViewEntry entry = null, nextEntry = null;
		ViewNavigator nav = null;
		try {
			nav=view.createViewNav();
			entry = nav.getFirst();
			while (entry != null) {
				if (entry.isDocument()) {
					String title = entry.getDocument().getItemValueString("F_titel");
					if (title.equalsIgnoreCase(key)) {
						return (entry);
					}
				}
				if (entry.isCategory()) {
					String category = entry.getColumnValues().elementAt(0).toString();
					if (category.equalsIgnoreCase(key)) {
						return (entry);
					}					
				}
				nextEntry = nav.getNext();
				GC.recycle(entry);
				entry=nextEntry;
			}
		} catch (NotesException e) {
			e.printStackTrace();
		} finally {
			GC.recycle (nav);
		}
		GC.recycle (entry);
		GC.recycle (nextEntry);
		return null;
	}
	
	/**
	 * Ldt ein Dokument mit Titel title. Es werden nur Dokumente
	 * mit Form == FO_dokument_k6 bercksichtigt.
	 * @param title
	 * @return
	 * @throws NotesException
	 */
	private static final Document getDocument (String title) throws NotesException {
		View lookup = null;
		Document myDoc = null;
		try {
			lookup = db.getView("V_lookup_k6");
			myDoc = lookup.getDocumentByKey(title.toLowerCase());
			return myDoc;
		} finally {
			if (lookup!=null) {
				lookup.recycle();
			}
		}
	}
	
	/**
	 * Diese Demo erzeugt sich in Main eigene Dokumente, um gezielt das Verhalten in
	 * einem View anzeigen zu knnen.
	 * Cleanup lscht diese Dokumente wieder.
	 * Bedingung ist, dass die Dokumente das Item F_marker_k6 mit Wert 1 enthalten.
	 * @throws NotesException
	 */
	private static final void cleanup () throws NotesException {
		DocumentCollection col = null;
		try {
			col = db.search ("F_marker_k6=1");
			System.out.println ("Aufrumen: Lsche " + col.getCount() + " Dokumente.");
			col.removeAll(true);
		} finally {
			if (col!=null) {
				col.recycle();
			}
		}
		
	}
	private static Database db = null;
	private static final String LINE = "********************************************************\n";
	private static final String[] SUB_ENTRIES = {"Erste Kategorie","Sub eins","1. Erstes Dokument"};
	private static final String[] SUB_DOCS = {"1. Erstes Dokument","2. Zweites Dokument","6. Sechstes Dokument"};
	private static final String [] SUB_CATEGORIE = {"Erste Kategorie", "Zweite Kategorie", "Zweite Kategorie\\Sub eins", "Sub eins"};

	private static final int TYPE_FROM = 0;
	private static final int TYPE_CHILDREN = 1;
	private static final int TYPE_DESCENDANTS = 2;

}
